/*
** Copyright 1998 - 2001 Double Precision, Inc.
** See COPYING for distribution information.
*/

#include	"courier.h"
#include	"localstatedir.h"
#include	"rw.h"
#include	"comcargs.h"
#include	"authlib/auth.h"
#include	"rfc822.h"
#include	"rfc1035/rfc1035.h"
#include	"rfc1035/rfc1035mxlist.h"
#include	"numlib/numlib.h"
#include	"dbobj.h"
#include	"afx.h"
#include	"aliases.h"
#include	"submit.h"
#include	"maxlongsize.h"
#include	<string.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<iostream>
#if	HAVE_LOCALE_H
#include	<locale.h>
#endif
#if	HAVE_SYS_TYPES_H
#include	<sys/types.h>
#endif
#if	HAVE_SYS_STAT_H
#include	<sys/stat.h>
#endif
#if	HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	<time.h>

#define	TIMEOUT	1800

static const char rcsid[]="$Id: submit.C,v 1.23 2001/05/03 02:09:57 mrsam Exp $";

time_t	submit_time;
static const char *expnname=0;
static const char *vrfyname=0;
static const char *receivedfrommta=0;
static const char *identinfo=0;
static const char *onlybcc=0;

const char *submitdelay=0;

static int nofilter;

int verpflag;
CString security;
extern CString	mkmessageidheader();

static struct courier_args arginfo[]={
	{"expn", &expnname},
	{"vrfy", &vrfyname},
	{"bcc", &onlybcc},

	{"delay", &submitdelay},
	{0}
	} ;

static const char you_are_whitelisted[]=
	"412 You are whitelisted by this recipient, please try again later.";

static const char you_are_not_whitelisted[]=
	"412 You are not whitelisted by this recipient, please try again later.";

// A pointer to expninfo is supplied as a misc argument to the
// rewrite_address function, which passes it along to the handler of the
// rewritten address.
//
// expninfo is used by the handler function (showexpn, showvrfy, or ...)
// in order to find the original AliasSearch function to search for
// any aliases of this address.

struct expninfo {
	AliasSearch *asptr;
	int	flag;
	struct rw_transport *module;
	} ;

/////////////////////////////////////////////////////////////////////////////
//
// rewrite_address() rewrites an arbitrary address using a specified module's
// rewriting function.
//
// inaddress - address to rewrite
// module    - module whose rewrite function is used
// mode      - rewriting mode
// sender    - what to put in rw_info.sender
//
// handler   - function that will be called.  The first argument to the
//             function will be the rewritten address, the second argument
//             is the same address, but as rfc822token link list.  The third
//             argument will be an error message.  If NULL, the address was
//             rewritten succesfully.  The last argument will be the misc
//             pointer.
// misc      - miscellaneous pointer to pass to the handler function.
//
/////////////////////////////////////////////////////////////////////////////

static void rewrite_address(const char *inaddress, struct rw_transport *module,
	int mode,
	struct rfc822token *sender,
	void (*handler)(const char *inaddress,
		struct rfc822token *inaddresst, const char *errmsg, void *misc),
	void *misc)
{
struct	rw_info_rewrite	rwir;
struct	rw_info rwi;
struct	rfc822t *rfcp;
char	*address, *p, *errmsg;

	rfcp=rw_rewrite_tokenize(inaddress);

	rw_info_init(&rwi, rfcp->tokens, rw_err_func);

	rwi.mode=mode;
	rwi.sender=sender;
	rwi.udata=(void *)&rwir;
	rwir.buf=0;
	rwir.errmsg=0;

	rw_rewrite_module(module, &rwi, rw_rewrite_chksyn_print);

	address=((struct rw_info_rewrite *)rwi.udata)->buf;
	if (address)
	{
	char	*hostdomain=0;

		if ((p=strrchr(address, '@')) &&
			config_islocal(p+1, &hostdomain) && hostdomain == 0)
			locallower(address);
		if (hostdomain)	free(hostdomain);
		domainlower(address);
	}
	errmsg=((struct rw_info_rewrite *)rwi.udata)->errmsg;

	(*handler)( address, rwi.ptr, errmsg, misc);
	if (address)	free(address);
	if (errmsg)	free(errmsg);
	rfc822t_free(rfcp);
}

int checkfreespace(unsigned *availbytes)
{
static unsigned nsubmitters;
static unsigned min_inodes=0;
unsigned long nfreeblocks, nfreeinodes;
unsigned blksize;
unsigned dummy;

	if (min_inodes == 0)	// Calculate thresholds
	{
	const char *p=getenv("MAXSUBMITTERS");

		nsubmitters= p ? (unsigned)atoi(p):0;

		if (nsubmitters < 10)	nsubmitters=10;
		if (nsubmitters > 1000)	nsubmitters=1000;

		min_inodes=nsubmitters*5+100;
	}

	if (!availbytes)	availbytes= &dummy;

	*availbytes=65536;

	if (freespace( ".", &nfreeblocks, &nfreeinodes, &blksize))
	{
		return (0);
	}

	if (nfreeinodes < min_inodes)	return (-1);

	if (nfreeblocks < blksize * nsubmitters/65536 + 100)
		*availbytes=4096;
	else if (nfreeblocks < blksize * nsubmitters/4096 + 100)
		return (-1);
	return (0);

}

//////////////////////////////////////////////////////////////////////
//
// ExpnAliasHandler - subclasses AliasHandler.  The AliasHandler
// class is used by the AliasSearch class to output addresses found
// in the aliases.dat file.  AliasHandler is subclassed so that
// one alias is always buffered.  When an alias address is received,
// the previous one is printed on cout, as an RFC822 multiline
// response.  The final call to ExpnAliasHandler::Alias() flushes out
// the final address and prints it as an RFC822 final multiline
// response.
//
//////////////////////////////////////////////////////////////////////

class ExpnAliasHandler : public AliasHandler {
public:
	ExpnAliasHandler();
	~ExpnAliasHandler();

	CString	prev_alias;
	void	Alias(const char *);
} ;

ExpnAliasHandler::ExpnAliasHandler()
{
}

ExpnAliasHandler::~ExpnAliasHandler()
{
}

void	ExpnAliasHandler::Alias(const char *p)
{
	if ( (const char *)prev_alias )
		cout << "250" << ( p ? '-':' ') <<
			"<" << prev_alias << ">" << endl << flush;

	if (p)
		prev_alias=p;
}

////////////////////////////////////////////////////////////////////////////
//
// showexpn - expand aliases.  After rewriting an address, expand any aliases.
//
////////////////////////////////////////////////////////////////////////////

static void showexpn(const char *address, struct rfc822token *addresst,
	const char *errmsg, void *misc)
{
struct	expninfo *info = (struct expninfo *)misc;

	addresst=addresst;

	if (errmsg)
	{
		cout << errmsg << flush;
		info->flag=1;
		return;
	}

	ExpnAliasHandler expnh;
	int rc;

	rc=info->asptr->Search(address, expnh);

	if (rc < 0)
		cout << "400 <" << address <<
			"> - service temporarily unavailable."
		     << endl << flush;
	else if (rc)
		cout << "500 <" << address << "> not found." << endl << flush;
				// No aliases
	else
		expnh.Alias(0);
				// Print last alias
	info->flag=0;
}

/////////////////////////////////////////////////////////////////////////////
//
// VRFY
//
// A pointer to rw_info_vrfy is passed to rw_searchdel as a misc pointer.
// The found_transport function is called when an output module is found
// for the address.  found_transport() saves the delivery information in
// the rw_info_vrfy structure
//
/////////////////////////////////////////////////////////////////////////////

struct rw_info_vrfy {
	struct rw_info_rewrite rwr;
	struct	rw_transport *module;
	char	*host;
	char	*addr;
	} ;

static void found_transport(struct rw_info *rwi,
		struct rw_transport *t,
		const struct rfc822token *host,
		const struct rfc822token *addr)
{
struct rw_info_vrfy *p=(struct rw_info_vrfy *)rwi->udata;

	p->module=t;
	if (!(p->host=rfc822_gettok(host)) ||
		!(p->addr=rfc822_gettok(addr)))
		clog_msg_errno();
}

static void showvrfy(const char *address, struct rfc822token *addresst,
	const char *errmsg, void *misc)
{
struct	expninfo *info = (struct expninfo *)misc;

	addresst=addresst;

	if (!address)
	{
		cout << errmsg << flush;
		info->flag=1;
		return;
	}

	// If there's an alias defined for this one, presume that this is
	// a good address.

	if (info->asptr->Found(address))
	{
		cout << "250 <" << address << ">" << endl << flush;
		return;
	}

	// Attempt to find an output module for this address.

struct	rfc822t *rfcp=rw_rewrite_tokenize(address);
struct	rw_info_vrfy riv;

	riv.rwr.buf=0;
	riv.rwr.errmsg=0;
	riv.module=0;
	riv.host=0;
	riv.addr=0;

struct	rw_info	rwi;

	rw_info_init(&rwi, rfcp->tokens, rw_err_func);
	rwi.udata= (void *)&riv;
	rwi.mode=RW_VERIFY;
	rw_searchdel(&rwi, &found_transport);
	rfc822t_free(rfcp);

	if (!riv.module && !riv.rwr.errmsg)
		rw_err_func(511, "Undeliverable address.", &rwi);


	if (!riv.module)	// Not found.
	{
		cout << riv.rwr.errmsg << flush;
		((struct expninfo *)misc)->flag=1;
		free(riv.rwr.errmsg);
		return;
	}
	free(riv.host);
	free(riv.addr);
	cout << "250 <" << address << ">" << endl << flush;
}

static void getrcpts(struct rw_info *rwi);

struct mailfrominfo {
	AliasSearch	*asptr;
	CString		*envid;
	char	dsnformat;
	struct	rw_transport *module;
	int	flag;
	} ;

static void mailfromerr(int msgnum, const char *errtext, struct rw_info *rwi)
{
	rwi=rwi;

char	*errmsg=makeerrmsgtext(msgnum, errtext);

	cout << errmsg << flush;
	free(errmsg);
}

static void setmsgopts(struct mailfrominfo &mfi, CString optstring)
{
	char *q=optstring.GetBuffer(-1);

	while (*q)
	{
		char ch= *q++;
		const char *opts=0;

		if (*q == '{')
		{
			opts= ++q;
			while (*q)
			{
				if (*q == '}')
				{
					*q++ = 0;
					break;
				}
				++q;
			}
		}

		switch (ch) {
		case 'f':
		case 'F':
			if (mfi.dsnformat == 0)
				mfi.dsnformat='F';
			break;
		case 'h':
		case 'H':
			if (mfi.dsnformat == 0)
				mfi.dsnformat='H';
			break;
		case 'V':
			verpflag=1;
			break;
		case 's':
		case 'S':
			if (opts && *opts && security.GetLength() == 0)
				security=opts;
			break;
		}
	}
	optstring.ReleaseBuffer(-1);
}

static int getsender(struct rw_transport *module)
{
struct	mailfrominfo frominfo;
CString	buf;
CString	sender;
CString	envid;
AliasSearch	asearch;

	frominfo.asptr= &asearch;
	frominfo.flag=1;
	frominfo.module=module;
	frominfo.dsnformat=0;
	frominfo.envid=0;

	if (buf << cin)	return (1);

char	*p=buf.GetBuffer(-1);
char	*q=strchr(p, '\t');

	if (q)
	{
		char *r;

		*q++=0;

		r=strchr(q, '\t');

		if (r)
			*r++=0;

		setmsgopts(frominfo, q);

		if (r && *r)
		{
			envid=r;
			frominfo.envid= &envid;
		}
	}
	if ((q=getenv("DSNRET")) != 0)
		setmsgopts(frominfo, q);

	buf.ReleaseBuffer(-1);
	sender=buf;

	if ((!frominfo.envid || !*frominfo.envid) &&
		(p=getenv("DSNENVID")) && *p)
	{
		for (envid=""; *p; p++)
		{
			if (*p < '!' || *p > '~' || *p == '+' ||
				*p == '=')
			{
			static char hexchar[17]="0123456789ABCDEF";

				envid += '+';
				envid += hexchar[ ((*p) >> 4) & 15];
				envid += hexchar[ (*p) & 15];
			}
			else	envid += *p;
		}
		frominfo.envid= &envid;
	}

struct	rfc822t *rfcp=rw_rewrite_tokenize(sender);
struct	rw_info	rwi;

	rw_info_init(&rwi, rfcp->tokens, mailfromerr);

	rwi.mode=RW_ENVSENDER|RW_SUBMIT;
	rwi.sender=NULL;
	rwi.udata=(void *)&frominfo;
	rwi.smodule=module->name;
	rw_rewrite_module(module, &rwi, getrcpts);
	rfc822t_free(rfcp);

	return (frominfo.flag);
}

static int checkdns(const char *sender)
{
struct	rfc1035_mxlist *mxlist, *p;
const char *h=0, *q;
char	*r;
int	docheckdomain=1, docheckbofh=0;

	q=getenv("BOFHCHECKDNS");
	if (!q || !*q || *q == '0')
		docheckdomain=0;

	if ((q=getenv("BOFHBANBROKENSERVERS")) != 0)
	{
		docheckbofh=atoi(q);
		h=courierdir();
	}

	if (!docheckdomain && !docheckbofh)	return (0);

	if ((sender=strrchr(sender, '@')) == 0)	return (0);
	++sender;

	switch (rfc1035_mxlist_create(&rfc1035_default_resolver,
		sender, &mxlist))	{
	case RFC1035_MX_OK:

		for (p=mxlist; p; p=p->next)
		{
		RFC1035_ADDR	addr;
		char	buf[RFC1035_NTOABUFSIZE];
		const	char *q;
		struct	stat stat_buf;

			if (rfc1035_sockaddrip(&p->address,
					sizeof(p->address), &addr))
				continue;

			rfc1035_ntoa(&addr, buf);
			if (strcmp(buf, p->hostname) == 0)
				break;
			if (docheckbofh <= 0)	continue;

			q=buf;
#if	RFC1035_IPV6
			if (IN6_IS_ADDR_V4MAPPED(&addr))
			{
				q=strrchr(q, ':');
				if (q)	++q;
				else q=buf;
			}
#endif
			r=(char *)courier_malloc(sizeof(TMPDIR "/broken/")+
				strlen(q));

			strcat(strcpy(r, TMPDIR "/broken/"), q);

			if (stat(r, &stat_buf) == 0 && stat_buf.st_size >=
				docheckbofh)
			{
				rfc1035_mxlist_free(mxlist);
				free(r);
				cout << "517-The mail server at IP address " << buf << " violates RFC 821." << endl << flush;
				cout << "517-Sorry, we cannot accept mail with a return address pointing to a broken" << endl << flush;
				cout << "517 mail server.  See <URL:ftp://ftp.isi.edu/in-notes/rfc821.txt>." << endl << flush;
				return (-1);
			}
			free(r);
		}
		rfc1035_mxlist_free(mxlist);
		if (p == 0)	return (0);
		cout << "517-MX records for " << sender << " violate section 3.3.9 of RFC 1035." << endl;
		break;
	case RFC1035_MX_HARDERR:
		if (!docheckdomain)	return (0);
		cout << "517-Domain does not exist: " << sender << "." << endl;
		break;
	case RFC1035_MX_BADDNS:
		if (!docheckdomain)	return (0);
		cout << "517-RFC 1035 violation: recursive CNAME records for " << sender << "." << endl;
		break;
	default:
		if (!docheckdomain)	return (0);
		cout << "417 DNS lookup failure: " << sender << ".  Try again later." << endl << flush;
		return (-1);
	}

	cout << "517 Invalid domain, see <URL:ftp://ftp.isi.edu/in-notes/rfc1035.txt>" << endl << flush;
	return (-1);
}

struct rcptinfo;

class RcptAliasHandler : public AliasHandler {
public:
	RcptAliasHandler();
	~RcptAliasHandler();
	void	Alias(const char *);
	SubmitFile *submitptr;
	struct rcptinfo *rcptinfoptr;
	CString	listname;
	unsigned listcount;
	CString	aliasbuf;

	CString	abortmsg;
	int aborted;
	int ret_code;

	struct rfc822token *addresst, *sendert;
	struct rw_transport *sourcemodule;
} ;

struct	rcptinfo {
	struct rw_transport *module;
	struct rfc822token *sendert;
	CString prw_receipient;	/* Pre-rewrite recipient */
	CString *oreceipient;	/* DSN original receipient */
	CString *dsn;		/* DSN flags */
	AliasSearch *asptr;
	int has_receipient;
	RcptAliasHandler rcptalias;
	SubmitFile submitfile;
	CString	errmsgtext;
	int	errflag;
	unsigned rcptnum;
	int	whitelisted_only;	/* Spam filter - allow whitelisted
					** recipients only.
					*/
	int	nonwhitelisted_only;	/* Spam filter - allow nonwhitelisted
					** recipients only.
					*/
	} ;

RcptAliasHandler::RcptAliasHandler()
{
}

RcptAliasHandler::~RcptAliasHandler()
{
}

static int do_receipient_filter(struct rw_info *, struct rw_info_vrfy *,
	CString &);

void RcptAliasHandler::Alias(const char *p)
{
	if (aborted)	return;

	if (listcount == 0)
	{
	struct rw_info	rwi;
	struct rw_info_vrfy riv;
	CString	errmsg;

		riv.rwr.buf=0;
		riv.rwr.errmsg=0;
		riv.module=0;
		riv.host=0;
		riv.addr=0;

		rw_info_init(&rwi, addresst, rw_err_func);
		rwi.sender=sendert;
		rwi.mode=RW_SUBMIT;
		rwi.udata= (void *)&riv;
		rwi.smodule=sourcemodule->name;

		riv.module=rw_search_transport("local");
		riv.rwr.errmsg=0;

	const char *cfa=config_filteracct();

		// Ok, we want the "host" portion to be
		// alias!alias-address!uid!gid!cfa!!

	int	rc=0;

		if (cfa && *cfa)
		{
		struct	stat	stat_buf;

			if (stat(cfa, &stat_buf))
			{
				aborted=1;
				abortmsg="400 stat() failed on aliasfilteracct.";
				return;
			}

		char	uidbuf[NUMBUFSIZE], gidbuf[NUMBUFSIZE];

			str_uid_t(stat_buf.st_uid, uidbuf);
			str_gid_t(stat_buf.st_gid, gidbuf);

	char	*fa=(char *)courier_malloc(strlen(cfa)+strlen(uidbuf)+
				strlen(gidbuf)+strlen(listname)+
				sizeof("alias!alias-!!!!!"));

			strcpy(fa, "alias!alias-");
			strcat(fa, listname);
			strcat(fa, "!");
			strcat(fa, uidbuf);
			strcat(fa, "!");
			strcat(fa, gidbuf);
			strcat(fa, "!");
			strcat(fa, cfa);
			strcat(fa, "!!");

			riv.host=fa;
			riv.addr="";	// Unused, for now

			rc=riv.module ?
				do_receipient_filter(&rwi, &riv, errmsg):0;
			free(fa);
		}

		ret_code=rc;

		if (rc)
		{
			if (rc != 99)
			{
				aborted=1;
				abortmsg=errmsg;
			}

			if (rcptinfoptr->whitelisted_only)
			{
				aborted=1;
				abortmsg=you_are_not_whitelisted;
			}
			else
				rcptinfoptr->nonwhitelisted_only=1;
		}
		else
		{
			if (rcptinfoptr->nonwhitelisted_only)
			{
				aborted=1;
				abortmsg=you_are_whitelisted;
			}
			else
				rcptinfoptr->whitelisted_only=1;
		}
			
		if (aborted)	return;

		aliasbuf=p;
		++listcount;
		return;
	}

	if (listcount == 1)
	{
		if (submitptr->ChkRecipient(aliasbuf) == 0)
			submitptr->AddReceipient(aliasbuf, "", "N", 0);
	}

	if (submitptr->ChkRecipient(p) == 0)
		submitptr->AddReceipient(p, "", "N", 0);
	++listcount;
}

static void getrcpt(struct rw_info *rwi);
static void rcpttoerr(int, const char *, struct rw_info *);

static void getrcpts(struct rw_info *rwi)
{
struct mailfrominfo *mf=(struct mailfrominfo *)rwi->udata;
struct rfc822token *addresst=rwi->ptr;
int	hasrcpts=0;
CString	impliciterrmsg;

	if (rw_syntaxchk(addresst))
	{
		cout << "517 Syntax error." << endl << flush;
		return;
	}

char	*sender=rfc822_gettok(addresst);

	if (!sender)	clog_msg_errno();

	if (checkdns(sender))	return;

	cout << "250 Ok." << endl << flush;

	mf->asptr->Open(mf->module);

struct	rcptinfo my_rcptinfo;
CString	receipient;
CString	oreceipient;
CString	dsn;

	my_rcptinfo.module=mf->module;
	my_rcptinfo.sendert=addresst;
	my_rcptinfo.asptr=mf->asptr;
	my_rcptinfo.has_receipient=0;
	my_rcptinfo.rcptalias.submitptr= &my_rcptinfo.submitfile;
	my_rcptinfo.rcptalias.rcptinfoptr= &my_rcptinfo;

	my_rcptinfo.submitfile.SendingModule( mf->module->name );

	my_rcptinfo.submitfile.Sender(receivedfrommta, sender,
			(mf->envid && mf->envid->GetLength() > 0 ?
			       (const char *)*mf->envid: (const char *)0),
			mf->dsnformat);
	my_rcptinfo.oreceipient= &oreceipient;
	my_rcptinfo.dsn= &dsn;
	my_rcptinfo.rcptnum=0;
	my_rcptinfo.whitelisted_only=0;
	my_rcptinfo.nonwhitelisted_only=0;

	free(sender);

	for (;; )
	{
		if (receipient << cin)	return;
		if (receipient.GetLength() == 0)	break;

	char	*p=receipient.GetBuffer(-1), *q;

		if ((p=strchr(p, '\t')) != NULL)
		{
			*p++=0;
			q=p;
			if ((p=strchr(p, '\t')) != NULL)
			{
				*p++=0;
				oreceipient=p;
			}
			else
				oreceipient="";
			dsn=q;
		}
		else
		{
			oreceipient="";
			dsn="";
		}
		receipient.ReleaseBuffer(-1);

	struct	rw_info rwi;
	struct	rfc822t *rfcp;

		rfcp=rw_rewrite_tokenize(receipient);
		rw_info_init(&rwi, rfcp->tokens, rcpttoerr);
		rwi.mode=RW_ENVRECIPIENT|RW_SUBMIT;
		rwi.sender=addresst;
		rwi.smodule=mf->module->name;
		rwi.udata=(void *)&my_rcptinfo;
		my_rcptinfo.errflag=0;
		my_rcptinfo.prw_receipient=receipient;

		rw_rewrite_module(mf->module, &rwi, getrcpt);

		// CANNOT reject if errflag=0 any more, due to filtering
		// syncronization.

		rfc822t_free(rfcp);
		hasrcpts=1;

		if (my_rcptinfo.errflag > 0)
		{
		const char *errmsg=my_rcptinfo.errmsgtext;
		int	l=atoi(errmsg);

			if (l < 400 || l > 599)	l=511;
			mailfromerr(l, errmsg, 0);
		}
		else
		{
			cout << "250 Ok." << endl << flush;
			my_rcptinfo.has_receipient=1;
			++my_rcptinfo.rcptnum;
		}
	}

	if (hasrcpts)
	{
		if (!my_rcptinfo.has_receipient)	return;
	}

	// Read headers

CString	header, headername, headernameorig;
CString	line;
CString	accumulated_errmsg;
int	has_msgid=0;
int	has_date=0;
int	has_tocc=0;
size_t	headercnt=500;
int	headerlimit=100000;
const	char *p;

	my_rcptinfo.submitfile.MessageStart();
	line="Received: from ";

	for (p=strchr(receivedfrommta, ';'); *++p == ' '; )
		;

	line += p;

	if (identinfo && *identinfo)
	{
		line += "\n  (";
		line += identinfo;
		line += ')';
	}

	line += "\n  by ";
	line += config_me();

	line += " with ";

	line += mf->module->name;
	line += "; ";
	line += rfc822_mkdate(submit_time);
	line += "\n";
	my_rcptinfo.submitfile.Message(line);

unsigned	received_cnt=0;

	while (line.readline(cin, 5000) > 0)
	{
	struct	rfc822t *rfcp;
	struct	rfc822a	*rfca;
	int	i;
	int	l;
	int	headerl;

		if ((i=cin.get()) != EOF && i != '\n')
		{
			headercnt=0;
			continue;	// Swallow unwanted message
		}

		l=line.GetLength();

		if (l && line[l-1] == '\r')
			line.Chop();	// Strip trailing CR

		header=line;

		while ( ((i=cin.get()) != EOF ? (cin.putback(i), i):i) == ' '
			|| i == '\t')
		{
			line.readline(cin, 5000);
			if ((i=cin.get()) != EOF && i != '\n')
			{
				headercnt=0;
				continue;	// Swallow unwanted message
			}
			l=line.GetLength();
			if (l && line[l-1] == '\r')
				line.Chop();	// Strip trailing CR
			l=line.GetLength() + header.GetLength();
			if (l > headerlimit || l > 5000)
				headercnt=0;
			if (headercnt == 0)	continue;
			header += '\n';
			header += line;
			if (i == EOF)	break;
		}
		header += '\n';
		if (header.GetLength() > headerlimit)
			headercnt=0;

		if (headercnt == 0 || --headercnt == 0)	continue;
		headerlimit -= (headerl=header.GetLength());

		i=header.Find(':');

		if (i < 0)
		{
			my_rcptinfo.submitfile.Message(header);
			continue;
		}

		headernameorig=headername=header.Left(i);
		header=header.Mid(i+1);
		headername.MakeLower();

		if (headername == "received")	++received_cnt;

		//
		// If no receipient were listed, grab them from the header.
		//

		if (!hasrcpts && (headername == "bcc" ||
			(onlybcc == 0 && (
			headername == "to" ||
			headername == "cc"))))
		{
			rfcp=rw_rewrite_tokenize(header);
			rfca=rfc822a_alloc(rfcp);
			if (!rfca)	clog_msg_errno();

			for (i=0; i<rfca->naddrs; i++)
			{
				if (rfca->addrs[i].tokens == NULL)
					continue;

			struct	rw_info rwi;

				rw_info_init(&rwi, rfca->addrs[i].tokens,
								rcpttoerr);
				rwi.mode=RW_ENVRECIPIENT|RW_SUBMIT;
				rwi.smodule=mf->module->name;
				rwi.sender=addresst;
				rwi.udata=(void *)&my_rcptinfo;
				my_rcptinfo.errflag=0;
				(*my_rcptinfo.oreceipient) = "";
				(*my_rcptinfo.dsn) = "";

			char	*p=rfc822_gettok(rfca->addrs[i].tokens);
				my_rcptinfo.prw_receipient=p;
				free(p);

				rw_rewrite_module(mf->module, &rwi, getrcpt);

		// CANNOT reject if errflag=0 any more, due to filtering
		// syncronization.

				if (my_rcptinfo.errflag > 0)
				{
					accumulated_errmsg +=
						my_rcptinfo.errmsgtext;
				}
				else
				{
					my_rcptinfo.has_receipient=1;
					++my_rcptinfo.rcptnum;
				}
			}
			rfc822a_free(rfca);
			rfc822t_free(rfcp);
		}

		// Rewrite every RFC822 header we can find.

		if (headername == "bcc")
		{
					// ... except this one

			headerlimit += headerl;
			++headercnt;
			continue;
		}

	int	istocc=0;

		if (headername == "to" || headername == "cc")
		{
			istocc=1;
			has_tocc=1;
		}

		if (istocc || headername == "from" ||
			headername == "reply-to" ||
			headername == "sender")
		{
		char	*errmsg;
		char	*new_header=rw_rewrite_header(mf->module,
				(const char *)header,
				RW_HEADER|RW_SUBMIT, addresst, &errmsg);

			if (!new_header)
			{
				accumulated_errmsg += errmsg;
				free(errmsg);
			}
			else
			{
			const char *p;
			char	*q;

				header="";
				p=" ";
				for (q=new_header; (q=strtok(q, "\n")) != 0;
					q=0)
				{
					header += p;
					header += q;
					p="\n    ";
				}
				free(new_header);
				header += '\n';
			}
		}
		else if (headername == "message-id")	has_msgid=1;
		else if (headername == "date")	has_date=1;

		// Quote Return-Path:'s at this point

		if (headername == "return-path")
			headernameorig=">"+headernameorig;

		my_rcptinfo.submitfile.Message(headernameorig);
		my_rcptinfo.submitfile.Message(":");
		my_rcptinfo.submitfile.Message(header);
	}

	if (!my_rcptinfo.has_receipient)
	{
		accumulated_errmsg += "511 Headers specify no receipients.\n";
	}

	if (accumulated_errmsg)
	{
	int	n=atoi(accumulated_errmsg);

		if ( (n / 100) != 4 && (n / 100) != 5)	n=511;

		while (cin.get() != EOF)
			;

	char	*buf=makeerrmsgtext(n, accumulated_errmsg);
		cout << buf << flush;
		free(buf);
		mf->flag=1;
		return;
	}

	if (received_cnt > 100)
		headercnt=0;

	if (headercnt)
	{
		if (!has_msgid)
		{
		const char *p=getenv("NOADDMSGID");

			if (!p || *p == '0')
				my_rcptinfo.submitfile.Message(
					mkmessageidheader());
		}

		if (!has_date)
		{
		const char *p=getenv("NOADDDATE");

			if (!p || *p == '0')
			{
				header="Date: ";
				header += rfc822_mkdate(submit_time);
				header += '\n';
				my_rcptinfo.submitfile.Message(header);
			}
		}
		if (!has_tocc)
			my_rcptinfo.submitfile.Message(
				"To: undisclosed-recipients: ;\n");
	}

	size_t bytecnt=0;

	while (!cin.eof())
	{
	int	i, l;

		line.readline(cin, 5000);
		if ( (i=cin.get()) != EOF && i != '\n')
			headercnt=0;
		if (headercnt == 0)	continue;
		l=line.GetLength();
		if (l && line[l-1] == '\r')
			line.Chop();	// Strip trailing CR

		if (i != EOF || line.GetLength())
		{
			line += '\n';
			my_rcptinfo.submitfile.Message(line);
		}
		if (i == EOF)	break;

		// Periodically reset the timeout

		if ( (bytecnt += l) >= 10000)
		{
			bytecnt=0;
			alarm(TIMEOUT);
		}
	}

	if (received_cnt > 100)
	{
		cout << "546 Routing loop detected -- too many Received: headers." << endl << flush;
		mf->flag=1;
	}
	else if (headercnt == 0)
	{
		cout << "534 Message header size, or recipient list, exceeds policy limit."
							<< endl << flush;
		mf->flag=1;
	}
	else
	{
		alarm(0);	/* Cancel alarm(1800) -- we'll wrap up soon */
		mf->flag=my_rcptinfo.submitfile.MessageEnd(my_rcptinfo.rcptnum,
			my_rcptinfo.whitelisted_only);
	}
}

static void rcpttoerr(int msgnum, const char *errtext, struct rw_info *rwi)
{
struct rcptinfo *my_rcptinfo=(struct rcptinfo *)rwi->udata;

char	*errmsg=makeerrmsgtext(msgnum, errtext);

	if (errmsg)
	{
		my_rcptinfo->errflag=1;
		my_rcptinfo->errmsgtext=errmsg;
		free(errmsg);
	}
}

static CString checkrcpt(struct rcptinfo *,
	struct rfc822token *,
	const char *);

static void getrcpt(struct rw_info *rwi)
{
struct rcptinfo *my_rcptinfo=(struct rcptinfo *)rwi->udata;
struct rfc822token *addresst=rwi->ptr;
char	*address=rfc822_gettok(addresst);

	if (!address)
		clog_msg_errno();

CString	errmsg;

	domainlower(address);
	errmsg=checkrcpt(my_rcptinfo, addresst, address);
	my_rcptinfo->errflag=0;

	if (errmsg.GetLength())
	{
		my_rcptinfo->errmsgtext=errmsg;
		my_rcptinfo->errflag=1;
	}
	free(address);
}

static CString checkrcpt(struct rcptinfo *my_rcptinfo,
	struct rfc822token *addresst,
	const char *address)
{
struct rw_transport *source=my_rcptinfo->module;
struct rfc822token *sendert=my_rcptinfo->sendert;
const char *oaddress=*my_rcptinfo->oreceipient;
const char *dsn=*my_rcptinfo->dsn;
AliasSearch &aliasp=*my_rcptinfo->asptr;
RcptAliasHandler &handlerp=my_rcptinfo->rcptalias;
unsigned rcptnum=my_rcptinfo->rcptnum;
const char *prw_receipient= my_rcptinfo->prw_receipient;

CString	errmsg;
char	envdsn[5];
int	dsnN, dsnS, dsnF, dsnD;
char	*addressl;
CString oaddressbuf;


char	deliv_envdsn[5];
CString	deliv_addressl;
CString	deliv_oaddress;

	if (!oaddress)	oaddress="";
	if (!dsn)	dsn="";

	strcpy(deliv_envdsn, "N");
	dsnN=0;
	dsnS=0;
	dsnF=0;
	dsnD=0;

	if (!*dsn && (dsn=getenv("DSNNOTIFY")) != 0)
		while (*dsn)
		{
			switch (*dsn)	{
			case 'n':
			case 'N':
				dsnN=1;
				break;
			case 's':
			case 'S':
				dsnS=1;
				break;
			case 'f':
			case 'F':
				dsnF=1;
				break;

			case 'd':
			case 'D':
				dsnD=1;
				break;
			}
			while (*dsn && *dsn != ',' && *dsn != ' ')
				++dsn;
			if (*dsn)	++dsn;
		}
	else if (dsn)
		for ( ; *dsn; dsn++)
		{
			switch (*dsn)	{
			case 'n':
			case 'N':
				dsnN=1;
				break;
			case 's':
			case 'S':
				dsnS=1;
				break;
			case 'f':
			case 'F':
				dsnF=1;
				break;

			case 'd':
			case 'D':
				dsnD=1;
				break;
			}
		}

	strcpy(envdsn, "");

	if (dsnN)
		strcpy(envdsn, "N");
	else
	{
		if (dsnS) strcat(envdsn, "S");
		if (dsnF) strcat(envdsn, "F");
		if (dsnD) strcat(envdsn, "D");
	}

	handlerp.listname=address;	// Just in case
	handlerp.listcount=0;
	handlerp.aborted=0;
	handlerp.ret_code=0;
	handlerp.addresst=addresst;
	handlerp.sendert=sendert;
	handlerp.sourcemodule=source;

	if (checkfreespace(0))
	{
		errmsg="431 Mail system full.";
		return (errmsg);
	}

	{
	const char *p=getenv("BLOCK");

		if (p && *p)
		{
			errmsg=p;
			return (errmsg);
		}
	}

	addressl=strcpy((char *)courier_malloc(strlen(address)+1), address);
	locallower(addressl);

	int search_rc=aliasp.Search(addressl, handlerp);

	if (search_rc < 0)
	{
		handlerp.abortmsg="400 Service temporarily unavailable.";
		return (handlerp.abortmsg);
	}

	if (search_rc ||
		handlerp.listcount == 1 ||
		handlerp.aborted)
	{
	struct rw_info	rwi;
	struct rw_info_vrfy riv;
	struct rfc822t *aliasaddresst=0;
	int	rwmode=RW_SUBMIT;

		if (handlerp.aborted)
		{
			return (handlerp.abortmsg);
		}

		/*
		** For aliases of exactly one address, we just replace
		** addresst with the tokenized alias.
		**
		** Note - if original address is not set, it is set to
		** the alias address.
		*/

		if (handlerp.listcount == 1)
		{
			rwmode |= RW_SUBMITALIAS;

			if (verpflag && strcmp(envdsn, "N"))

			/* This is effectively a rewrite of a VERPed
			** recipient, so the buck stops here.
			*/
			{
				strcpy(deliv_envdsn, envdsn);
				deliv_addressl=addressl;
				deliv_oaddress=oaddress;
					/* Generate a "delivered" notice if
					** a DSN was requested */

				strcpy(envdsn, "N");
					/* No errors for this one */
			}

			aliasaddresst=rw_rewrite_tokenize(handlerp.aliasbuf);
			if (!aliasaddresst)	clog_msg_errno();

			if (*oaddress == 0)
			{
			char	*p=dsnencodeorigaddr(address);

				oaddressbuf=p;
				free(p);
				oaddress=oaddressbuf;
			}

			addresst=aliasaddresst->tokens;
			address=handlerp.aliasbuf;
		}

		/*
		**  If the module functions rewrote the recipient address, and
		**  the verp flag is set, we need to make sure to generate a
		**  DSN for final delivery, and suppress any more DSNs for
		**  this recipient, from now on, because the sender will not
		**  have any idea that the address has been aliased.
		*/

		else if (verpflag && strcmp(address, prw_receipient) &&
			strcmp(envdsn, "N"))
		{
			strcpy(deliv_envdsn, envdsn);
			deliv_addressl=addressl;
			deliv_oaddress=oaddress;
			strcpy(envdsn, "N");
		}
		free(addressl);

		riv.rwr.buf=0;
		riv.rwr.errmsg=0;
		riv.module=0;
		riv.host=0;
		riv.addr=0;

		rw_info_init(&rwi, addresst, rw_err_func);
		rwi.sender=sendert;
		rwi.mode=rwmode;
		rwi.udata= (void *)&riv;
		rwi.smodule=source->name;

		/* Just let the DSN source through */

		if (strcmp(source->name, DSN) == 0)
		{
			riv.module=source;
			riv.rwr.errmsg=0;
			riv.host=strcpy((char *)courier_malloc(1), "");
			riv.addr=strcpy((char *)courier_malloc(
				strlen(address)+1), address);
		}
		else
		{
			if (!nofilter)	
				rwi.mode |= RW_FILTER;
			rw_searchdel(&rwi, &found_transport);
		}

		if (aliasaddresst)
			rfc822t_free(aliasaddresst);

		if (!riv.module && !riv.rwr.errmsg)
			rw_err_func(511, "Unknown user.", &rwi);
		if (!riv.module)
		{
			errmsg=riv.rwr.errmsg;
			free(riv.rwr.errmsg);
			return (errmsg);
		}

		/* If we're here because of a one-alias recipient, pretend
		** that it's a whitelisted recipient */
		
	int	rc=handlerp.listcount == 1 ? handlerp.ret_code:
			do_receipient_filter(&rwi, &riv, errmsg);

		if (rc)
		{
			if (rc != 99)	return (errmsg);

			/*
			** Return code 99 -- nonwhitelisted recipient subject
			** to spam filtering.
			*/
			if (my_rcptinfo->whitelisted_only)
			{
				errmsg=you_are_not_whitelisted;
				return (errmsg);
			}
			my_rcptinfo->nonwhitelisted_only=1;
		}
		else
		{
			/*
			** Return code 0 - whitelisted recipient.
			*/
			if (my_rcptinfo->nonwhitelisted_only)
			{
				errmsg=you_are_whitelisted;
				return (errmsg);
			}
			my_rcptinfo->whitelisted_only=1;
		}

		// Good address, but should we add it to the list of
		// recipients?  Figure out what should be the key to check
		// for a duplicate address.
		//
		// DSN: NEVER, use module<nl>host<nl>addr, should get rid
		//      of the most number of duplicate addresses
		//
		// Every case, use module<nl>host<nl>addr<nl>orecipient.

	CString key;
	int	dupflag=0;

		key=riv.module->name;
		key += '\n';
		key += riv.host;
		key += '\n';
		key += riv.addr;
		if (strcmp(envdsn, "N") == 0)
		{
			if (handlerp.submitptr->ChkRecipient(key))
				dupflag=1;
		}

		key += '\n';
		key += oaddress;

		if (handlerp.submitptr->ChkRecipient(key))
			dupflag=1;

		if (!dupflag)
		{
			if (strcmp(deliv_envdsn, "N"))
				handlerp.submitptr->AddReceipient(
					deliv_addressl,
					deliv_oaddress,
					deliv_envdsn, 1);

			handlerp.submitptr->AddReceipient(address,
					oaddress, envdsn, 0);

			/*
			** If we're handling nonwhitelisted recipients,
			** record their control file location for the
			** spam filter, to follow later.
			*/

			if (my_rcptinfo->nonwhitelisted_only
				&& handlerp.listcount != 1) /* NOT ALIASES */
				handlerp.submitptr->ReceipientFilter(
					riv.module,
					riv.host,
					riv.addr,
					rcptnum);
		}

		free(riv.host);
		free(riv.addr);
	}
	else	// Make sure aliases are properly recorded for DSN generation
		// purposes.
	{
		handlerp.submitptr->AddReceipient(addressl, oaddress, envdsn,1);
		free(addressl);
	}
	return (errmsg);
}

static int do_receipient_filter(struct rw_info *rwi, struct rw_info_vrfy *riv,
	CString &errmsg)
{
int	rc;
char	errmsgbuf[2048];
char *p;

	if (riv->module->rw_ptr->filter_msg == 0)
		return (0);		/* Handle as a whitelisted recipient */

	if (nofilter)
		return (0);		/* Caller module is whitelisted */

	p=rfc822_gettok(rwi->sender);
	if (!p)	clog_msg_errno();

	errmsgbuf[0]=0;
	rc=(*riv->module->rw_ptr->filter_msg)(
		rwi->smodule,
		-1,
		riv->host,
		riv->addr,
		p,
		errmsgbuf,
		sizeof(errmsgbuf));
	free(p);

	if (!(rc == 0 || rc == 99))
	{
		p=errmsgbuf;
		if (!*p)
			p="571 Delivery not authorized, message refused.";
		errmsg=p;
	}
	return (rc);
}

int cppmain(int argc, char **argv)
{
int	argn;
const	char *module;

	clog_open_syslog("submit");
	umask(007);
#if HAVE_SETLOCALE
	setlocale(LC_ALL, "C");
#endif
	putenv("SHELL=/bin/sh");
	// In 0.34 maildrop environment init was changed to import SHELL,
	// when built as part of Courier.  Therefore, we need to make sure
	// that SHELL is initialized here.

	if (getuid() == 0)	/* Running as root screws up perms */
		authchangeuidgid(MAILUID, MAILGID);

	argn=cargs(argc, argv, arginfo);
	if (argc - argn < 2)
		exit(1);

	module=argv[argn];
	receivedfrommta=argv[argn+1];
	if (strchr(receivedfrommta, ';') == 0)	exit(1);

	if (argc - argn > 2)
		identinfo=argv[argn+2];

	if (chdir(courierdir()) || chdir(TMPDIR))	clog_msg_errno();

	if (rw_init(0))	exit (1);

struct rw_transport *modulep=rw_search_transport(module);
 
	if (!modulep)
	{
		clog_msg_start_err();
		clog_msg_str(module);
		clog_msg_str(" - not a valid module.");
		clog_msg_send();
		exit(1);
	}

	if (expnname || vrfyname)
	{
	struct	expninfo expn_info;
	AliasSearch	asearch;

		{

			asearch.Open(modulep);
			expn_info.asptr=&asearch;
			expn_info.flag=0;

			rewrite_address(expnname ? expnname:vrfyname,
				modulep,
				(expnname ? RW_ENVRECIPIENT|RW_EXPN:
					RW_ENVRECIPIENT|RW_VERIFY), NULL,
				expnname ? showexpn:showvrfy, &expn_info);
		}
		exit(expn_info.flag);
	}

	time(&submit_time);

	SubmitFile::trapsignals();
	verpflag=0;

	/* Determine if mail filtering should be used */

	nofilter=1;
	{
	char	*fn=config_localfilename("enablefiltering");
	char	*list=config_read1l(fn);

		free(fn);
		if (list)
		{
		char	*p;

			for (p=list; (p=strtok(p, " \r\t\n")) != 0; p=0)
				if (strcmp(p, module) == 0)
				{
					nofilter=0;
					break;
				}
			free(list);
		}
	}

	alarm(TIMEOUT);	/* Kill me, no matter what */
	exit (getsender(modulep));
	return (0);
}

extern "C" void rfc2045_error(const char *errmsg)
{
	cout << "430 " << errmsg << endl << flush;
	exit(1);
}
